//******************************************************************************
//  MSP430F20xx - I2C Master Transmitter and Receiver via CPU Software (GPIO)
//
//  Description: This code library configures the MSP430 as an I2C master device
//  capable of transmitting and receiving bytes using GPIO pins.  Specific I/O
//  pins can be selected in the corresponding header file.  By default, the same
//  pins that are used by the USI module are used in this example.
//
//                  Master                   
//                  MSP430          
//             -----------------          
//         /|\|              XIN|-   
//          | |                 |     
//          --|RST          XOUT|-    
//            |                 |        
//            |                 |        
//            |                 |       
//            |             Px.y|-------> [I2C SLAVE SDA]
//            |             Px.z|-------> [I2C SLAVE SCL]
//
//  Note: Internal pull-ups are NOT used for SDA & SCL [DISABLED]
//
//  R. Wu
//  Texas Instruments Inc.
//  January 2010
//
//******************************************************************************

//changes made: wait for clock stretching on transmit byte
//				add I2CDELAY before clock brought high for ACK on tx/rxbyteheader 

#include "MSP430_SWI2C_Master.h"

void MSP430_SWI2CMST_init(void)
{
  PxOUT &= ~(SCL | SDA);                    // Output settings always LOW
  PxSEL &= ~(SCL | SDA);                    // Set GPIO functions
}

void MSP430_SWI2CMST_delay(void)
{
  __delay_cycles(GPIODELAYCYCLES);          // Quick delay
}

void MSP430_SWI2CMST_start(void)
{
  SDA_1;                                    // SDA = 1
  I2CDELAY;                                 // Quick delay
  SCL_1;                                    // SCL = 1
  I2CDELAY;                                 // Quick delay
  SDA_0;                                    // SDA = 0
  I2CDELAY;                                 // Quick delay
  SCL_0;                                    // SCL = 0
  I2CDELAY;                                 // Quick delay  
}

void MSP430_SWI2CMST_stop(void)
{ 
  SDA_0;                                    // SDA = 0
  I2CDELAY;                                 // Quick delay
  SCL_1;                                    // SCL = 1
  I2CDELAY;                                 // Quick delay
  SDA_1;                                    // SDA = 1
  I2CDELAY;                                 // Quick delay 
}

unsigned char MSP430_SWI2CMST_txByte(unsigned char data)
{
  unsigned char bits, temp, ack;
  
  SCL_0;                                    // SCL = 0
  temp = data;                              // Initialize temp variable
  bits = 0x08;                              // Load I2C bit counter
  while (bits != 0x00)                      // Loop until all bits are shifted
  {
    if (temp & BIT7)                        // Test data bit
      SDA_1;                                // SDA = 1
    else
      SDA_0;                                // SDA = 0
    I2CDELAY;                               // Quick delay    
    SCL_1;                                  // SCL = 1
    while ((PxIN & SCL) == 0);              // Wait for any SCL clock stretching
    I2CDELAY;                               // Quick delay
    temp = (temp << 1);                     // Shift bits 1 place to the left
    SCL_0;                                  // SCL = 0
    bits = (bits - 1);                      // Loop until 8 bits are sent
  }
  I2CDELAY;
  SDA_1;                                    // SDA = 1
  SCL_1;                                    // SCL = 1
  while ((PxIN & SCL) == 0);              // Wait for any SCL clock stretching
  I2CDELAY;                                 // Quick delay
  ack = (PxIN & SDA);                       // Read ACK state from Slave
  SCL_0;                                    // SCL = 0
  if (ack)                                  // Return ACK state to calling app
    return (1);
  else
    return (0);
}

unsigned char MSP430_SWI2CMST_rxByte(char ack)
{
  unsigned char bits, data = 0;

  SDA_1;                                    // SDA = 1
  bits = 0x08;                              // Load I2C bit counter
  while (bits > 0)                          // Loop until all bits are read
  {
    SCL_1;                                  // SCL = 1
    while ((PxIN & SCL) == 0);              // Wait for any SCL clock stretching
    I2CDELAY;                               // Quick delay
    data = (data << 1);                     // Shift bits 1 place to the left
    if (PxIN & SDA)                         // Check digital input
      data = (data + 1);                    // If input is high, store a '1'
    SCL_0;                                  // SCL = 0
    I2CDELAY;                               // Quick delay
    bits = (bits - 1);                      // Decrement I2C bit counter
  }
  if (ack)                                  // Need to send ACK to Slave?
    SDA_0;                                  // Yes, so pull SDA low
  else
    SDA_1;                                  // No, so keep SDA high
  SCL_1;                                    // SCL = 1
  I2CDELAY;                                 // Equivalent to sending N(ACK)
  SCL_0;                                    // SCL = 0
  SDA_1;                                    // SDA = 1

  return (data);                            // Return 8-bit data byte
}   

void MSP430_SWI2CMST_writeBlock(unsigned char SlaveAddress,
                                unsigned int numBytes, unsigned char multi,
                                void* TxData)
{        
  unsigned int  i;
  unsigned char *temp;
   
  temp = (unsigned char *)TxData;           // Initialize array pointer
  MSP430_SWI2CMST_start();                  // Send Start condition
  MSP430_SWI2CMST_txByte((SlaveAddress << 1) & ~BIT0); // [ADDR] + R/W bit = 0
  for (i = 0; i < numBytes; i++)
  {
    MSP430_SWI2CMST_txByte(*(temp));        // Send data and ack
    temp++;                                 // Increment pointer to next element
  }
  if (multi == 0)                           // Need STOP condition?
  {
    MSP430_SWI2CMST_stop();                 // Yes, send STOP condition
  }
  I2CDELAY;                                 // Quick delay
}

void MSP430_SWI2CMST_readBlock(unsigned char SlaveAddress,
                               unsigned int numBytes, void* RxData)
{
  unsigned int  i;
  unsigned char* temp;
  
  temp = (unsigned char *)RxData;           // Initialize array pointer
  MSP430_SWI2CMST_start();                  // Send Start condition
  MSP430_SWI2CMST_txByte((SlaveAddress << 1) | BIT0); // [ADDR] + R/W bit = 1
  for (i = 0; i < numBytes; i++)
  {
    if (i == (numBytes - 1))
      *(temp) = MSP430_SWI2CMST_rxByte(NACK);// Read last 8-bit data with no ACK
    else
      *(temp) = MSP430_SWI2CMST_rxByte(ACK);// Read 8-bit data & then send ACK   
    temp++;                                 // Increment pointer to next element
  }
  MSP430_SWI2CMST_stop();                   // Send Stop condition
}
